import json, time, datetime, requests
from typing import List, Dict

try:
    from zk import ZK  # pyzk
except Exception as e:
    ZK = None

def load_config(path="agent_config.json") -> Dict:
    with open(path, "r", encoding="utf-8") as f:
        return json.load(f)

def pull_device_logs(ip: str, port: int) -> List[Dict]:
    if ZK is None:
        raise RuntimeError("pyzk not installed. Run: pip install pyzk")
    zk = ZK(ip, port=port, timeout=10, password=0, force_udp=False, ommit_ping=False)
    conn = zk.connect()
    try:
        records = conn.get_attendance()
        logs = []
        for r in records:
            # r.user_id, r.timestamp, r.status, r.punch
            logs.append({
                "uid": str(getattr(r, "user_id", "")),
                "timestamp": getattr(r, "timestamp", None).isoformat(sep=" ", timespec="seconds") if getattr(r, "timestamp", None) else "",
                "verify_type": str(getattr(r, "status", "")),
                "punch_type": str(getattr(r, "punch", "")),
            })
        return logs
    finally:
        try:
            conn.disconnect()
        except Exception:
            pass

def push_logs(cloud_url: str, branch_code: str, agent_key: str, device_serial: str, logs: List[Dict]) -> Dict:
    url = cloud_url.rstrip("/") + "/?page=api/v1/attendance_push"
    payload = {
        "branch_code": branch_code,
        "agent_key": agent_key,
        "device_serial": device_serial,
        "logs": logs,
    }
    r = requests.post(url, json=payload, timeout=25)
    try:
        return {"status": r.status_code, "json": r.json()}
    except Exception:
        return {"status": r.status_code, "text": r.text[:500]}

def main():
    cfg = load_config()
    cloud_url = cfg["cloud_url"]
    branch_code = cfg["branch_code"]
    agent_key = cfg["agent_key"]
    poll = int(cfg.get("poll_seconds", 60))
    devices = cfg.get("devices", [])

    print("Gustolix Attendance Sync Agent")
    print("Cloud:", cloud_url, "| Branch:", branch_code, "| Poll:", poll, "seconds")
    print("Devices:", len(devices))

    while True:
        for d in devices:
            name = d.get("name", "Device")
            ip = d["ip"]
            port = int(d.get("port", 4370))
            serial = d["serial_no"]

            try:
                logs = pull_device_logs(ip, port)
                # To reduce payload: send only last N logs each round (safe due to de-duplication)
                logs = logs[-500:]
                res = push_logs(cloud_url, branch_code, agent_key, serial, logs)
                print(datetime.datetime.now().isoformat(timespec="seconds"), "-", name, ip, "sent", len(logs), "->", res.get("status"), res.get("json") or res.get("text"))
            except Exception as e:
                print(datetime.datetime.now().isoformat(timespec="seconds"), "-", name, ip, "ERROR:", str(e)[:250])
        time.sleep(poll)

if __name__ == "__main__":
    main()
