<?php
declare(strict_types=1);

namespace App\Core;

use PDO;
use PDOException;

final class DB {
  private static ?PDO $pdo = null;

  public static function pdo(): PDO {
    if (self::$pdo) return self::$pdo;

    $host = Env::get('DB_HOST', 'localhost');
    $db   = Env::get('DB_NAME', '');
    $user = Env::get('DB_USER', '');
    $pass = Env::get('DB_PASS', '');

    $dsn = "mysql:host={$host};dbname={$db};charset=utf8mb4";
    try {
      self::$pdo = new PDO($dsn, $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
      ]);
    } catch (PDOException $e) {
      http_response_code(500);
      echo "<h1>DB connection failed</h1>";
      echo "<pre>" . htmlspecialchars($e->getMessage()) . "</pre>";
      exit;
    }

    return self::$pdo;
  }

  public static function one(string $sql, array $params=[]): ?array {
    $st = self::pdo()->prepare($sql);
    $st->execute($params);
    $r = $st->fetch();
    return $r ?: null;
  }

  public static function all(string $sql, array $params=[]): array {
    $st = self::pdo()->prepare($sql);
    $st->execute($params);
    return $st->fetchAll();
  }

  public static function exec(string $sql, array $params=[]): int {
    $st = self::pdo()->prepare($sql);
    $st->execute($params);
    return $st->rowCount();
  }

  public static function id(): string {
    return (string)self::pdo()->lastInsertId();
  }
}
